#-----------------------------------------------------------------------------#
# Copyright (C) 1999,2001-2004, 2006-2012 The University of Melbourne.
# Copyright (C) 2013-2025 The Mercury team.
# This file may only be copied under the terms of the GNU General
# Public Licence - see the file COPYING in the Mercury distribution.
#-----------------------------------------------------------------------------#
#
# aclocal.m4
#
# This file contains Mercury-specific autoconf tests.
#
# We ought to move most of the code in configure.in into this file...
#
#-----------------------------------------------------------------------------#
AC_DEFUN([MERCURY_CHECK_FOR_HEADERS],
[
    for mercury_cv_header in $1; do
	mercury_cv_header_define="MR_HAVE_`echo $mercury_cv_header | \
		tr abcdefghijklmnopqrstuvwxyz./ ABCDEFGHIJKLMNOPQRSTUVWXYZ__`"
	AC_CHECK_HEADER($mercury_cv_header, [
		AC_DEFINE_UNQUOTED($mercury_cv_header_define)
		eval "$mercury_cv_header_define=1"
	])
    done
])
#-----------------------------------------------------------------------------#
AC_DEFUN([MERCURY_CHECK_FOR_IEEEFP_H],
[
	MERCURY_CHECK_FOR_HEADERS(ieeefp.h)
])

AC_DEFUN([MERCURY_CHECK_FOR_IEEE_FUNC],
[
AC_REQUIRE([MERCURY_CHECK_FOR_IEEEFP_H])
AC_MSG_CHECKING(for $1 function)
mercury_cv_ieee_func_define="MR_HAVE_`echo $1 | \
	tr abcdefghijklmnopqrstuvwxyz./ ABCDEFGHIJKLMNOPQRSTUVWXYZ__`"

AC_LINK_IFELSE([AC_LANG_PROGRAM([[
	#include <math.h>
#ifdef MR_HAVE_IEEEFP_H
	#include <ieeefp.h>
#endif
]], [[
	float f;
	$1(f);
]])],[mercury_cv_have_ieee_func=yes],[mercury_cv_have_ieee_func=no])

if test "$mercury_cv_have_ieee_func" = yes; then
	AC_MSG_RESULT(yes)
	AC_DEFINE_UNQUOTED($mercury_cv_ieee_func_define)
else
	AC_MSG_RESULT(no)
fi
])

#-----------------------------------------------------------------------------#

# Test for C99 fp environment functions in the math library.
# The second argument to this macro should be the flags required by the
# C compiler to link against the math library. This is needed because
# some OSs, e.g. Mac OS X, don't have a separate math library.

AC_DEFUN([MERCURY_CHECK_FOR_FENV_FUNC],
[
AC_MSG_CHECKING(for $1 function)
mercury_cv_math_func_define="MR_HAVE_`echo $1 | \
	tr abcdefghijklmnopqrstuvwxyz./ ABCDEFGHIJKLMNOPQRSTUVWXYZ__`"

save_libs="$LIBS"
LIBS="$2 $LIBS"

AC_LINK_IFELSE([AC_LANG_PROGRAM([[
#ifdef MR_HAVE_FENV_H
	#include <fenv.h>
#endif
]], [[

	int i = 0;
	$1(i);
]])],[mercury_cv_have_math_func=yes],[mercury_cv_have_math_func=no])

LIBS="$save_libs"

if test "$mercury_cv_have_math_func" = "yes"
then
	AC_MSG_RESULT([yes])
	AC_DEFINE_UNQUOTED([$mercury_cv_math_func_define])
else
	AC_MSG_RESULT([no])
fi
])

#-----------------------------------------------------------------------------#
#
# Turn off MacOS's so-called "smart" C preprocessor, if present,
# since it causes lots of spurious warning messages,
# and furthermore it takes way too long and uses way too much memory
# when preprocessing the C code generated by the Mercury compiler's LLDS
# back-end.
#
AC_DEFUN([MERCURY_CHECK_CC_NEEDS_TRAD_CPP],
[
AC_REQUIRE([AC_PROG_CC])
AC_MSG_CHECKING(whether C compiler needs -no-cpp-precomp)
AC_CACHE_VAL(mercury_cv_cpp_precomp, [
	>conftest.c
	if	test "$GCC" = yes &&
		$CC -v -c conftest.c 2>&1 | \
			grep "cpp-precomp.*-smart" > /dev/null
	then
		mercury_cv_cpp_precomp=yes
	else
		mercury_cv_cpp_precomp=no
	fi
])
AC_MSG_RESULT($mercury_cv_cpp_precomp)
if test $mercury_cv_cpp_precomp = yes; then
	CC="$CC -no-cpp-precomp"
fi
])
#-----------------------------------------------------------------------------#
#
# Check whether we need to add any extra directories to the search path
# for header files, and set ALL_LOCAL_C_INCL_DIRS to the -I option(s) needed
# for this, if any.
#
# GNU C normally searches /usr/local/include by default;
# to keep things consistent, we do the same for other C compilers.
#
AC_DEFUN([MERCURY_CHECK_LOCAL_C_INCL_DIRS],
[
AC_REQUIRE([AC_PROG_CC])
AC_MSG_CHECKING(whether to pass -I/usr/local/include to C compiler)
ALL_LOCAL_C_INCL_DIRS=""
ALL_LOCAL_C_INCL_DIR_MMC_OPTS=""

if test "$GCC" = yes -o "$USING_MICROSOFT_CL_COMPILER" = yes; then
	# Don't add -I/usr/local/include, since it causes a warning
	# with gcc 3.1, and gcc already searches /usr/local/include.
	# Microsoft compilers don't understand Unix pathnames.
	AC_MSG_RESULT(no)
else
	# It is some other compiler. We don't know if it searches
	# /usr/local/include by default, so add it.
	if test -d /usr/local/include/.; then
		AC_MSG_RESULT(yes)
		ALL_LOCAL_C_INCL_DIRS="-I/usr/local/include "
		ALL_LOCAL_C_INCL_DIR_MMC_OPTS="--c-include-directory /usr/local/include "
	else
		AC_MSG_RESULT(no)
	fi
fi
AC_SUBST(ALL_LOCAL_C_INCL_DIRS)
AC_SUBST(ALL_LOCAL_C_INCL_DIR_MMC_OPTS)
])
#-----------------------------------------------------------------------------#
#
# Set ALL_LOCAL_C_LIB_DIRS to any extra directories we need to add
# to the search path for libraries.
#
AC_DEFUN([MERCURY_CHECK_LOCAL_C_LIB_DIRS],
[
AC_MSG_CHECKING(whether to pass -L/usr/local/lib to the linker)

# Microsoft compilers don't understand Unix pathnames.
if test "$USING_MICROSOFT_CL_COMPILER" = no -a -d /usr/local/lib/.; then
	AC_MSG_RESULT(yes)
	ALL_LOCAL_C_LIB_DIRS=/usr/local/lib
	ALL_LOCAL_C_LIB_DIR_MMC_OPTS="-L/usr/local/lib -R/usr/local/lib"
else
	AC_MSG_RESULT(no)
	ALL_LOCAL_C_LIB_DIRS=
	ALL_LOCAL_C_LIB_DIR_MMC_OPTS=
fi
AC_SUBST(ALL_LOCAL_C_LIB_DIRS)
AC_SUBST(ALL_LOCAL_C_LIB_DIR_MMC_OPTS)
])

#-----------------------------------------------------------------------------#
#
# Check if a C expression is true at compile time.
#

AC_DEFUN([MERCURY_TRY_STATIC_ASSERT], [
    AC_COMPILE_IFELSE([AC_LANG_PROGRAM([[$1]], [[
	    struct mercury_static_assert {
		int static_assert_expr : ( $2 ) ? 1 : -1;
	    };
	]])],[$3],[$4])
])

#-----------------------------------------------------------------------------#
#
# Check for readline and related header files and libraries.
#
AC_DEFUN([MERCURY_CHECK_READLINE],
[
AC_ARG_WITH(readline,
[  --without-readline      Do not use the GPL GNU readline library],
mercury_cv_with_readline="$withval", mercury_cv_with_readline=yes)

if test "$mercury_cv_with_readline" = yes; then

	# Check for the readline header files.
	MERCURY_CHECK_FOR_HEADERS(readline/readline.h readline/history.h)

	if test "$MR_HAVE_READLINE_READLINE_H,$MR_HAVE_READLINE_HISTORY_H" = "1,1"; then
		# Check for the libraries that readline depends on.
		MERCURY_MSG('looking for termcap or curses (needed by readline)...')
		AC_CHECK_LIB(termcap, tgetent, mercury_cv_termcap_lib=-ltermcap,
		 [AC_CHECK_LIB(curses,  tgetent, mercury_cv_termcap_lib=-lcurses,
		  [AC_CHECK_LIB(ncurses, tgetent, mercury_cv_termcap_lib=-lncurses,
		   mercury_cv_termcap_lib='')])])

		# Check for the readline library.
		AC_CHECK_LIB(readline, readline, mercury_cv_have_readline=yes,
		    mercury_cv_have_readline=no, $mercury_cv_termcap_lib)
	else
		mercury_cv_have_readline=no
	fi
else
	mercury_cv_have_readline=no
fi

if test $mercury_cv_have_readline = no; then
	TERMCAP_LIBRARY=""
	READLINE_LIBRARIES=""
else
	AC_DEFINE(MR_USE_READLINE)
	TERMCAP_LIBRARY="$mercury_cv_termcap_lib"
	READLINE_LIBRARIES="-lreadline $TERMCAP_LIBRARY"
fi
AC_SUBST(TERMCAP_LIBRARY)
AC_SUBST(READLINE_LIBRARIES)

])

#-----------------------------------------------------------------------------#

AC_DEFUN([MERCURY_CHECK_EDITLINE], [
AC_REQUIRE([MERCURY_CHECK_READLINE])
AC_ARG_WITH(editline,
[  --without-editline      Do not use the NetBSD editline library (libedit)],
mercury_cv_with_editline="$withval", mercury_cv_with_editline=yes)

if test $mercury_cv_with_editline = no; then
	mercury_cv_have_editline=no
elif test $mercury_cv_have_readline = no; then

	# Check for the editline header files.
	MERCURY_CHECK_FOR_HEADERS(editline/readline.h)

	if test "$MR_HAVE_EDITLINE_READLINE_H" = 1; then
		# Check for the editline library.
		AC_CHECK_LIB(edit, readline, mercury_cv_have_editline=yes,
			mercury_cv_have_editline=no)
	else
		mercury_cv_have_editline=no
	fi
else
	mercury_cv_have_editline=no
fi

if test $mercury_cv_have_editline = yes; then
	AC_DEFINE(MR_USE_EDITLINE)
	TERMCAP_LIBRARY=""
	READLINE_LIBRARIES="-ledit"
fi

])

#-----------------------------------------------------------------------------#
#
# Microsoft.NET configuration
#
AC_DEFUN([MERCURY_CHECK_DOTNET],
[
AC_PATH_PROG([ILASM], [ilasm])
AC_PATH_PROG([GACUTIL], [gacutil])

AC_MSG_CHECKING([for Microsoft.NET Framework SDK])
AC_CACHE_VAL([mercury_cv_microsoft_dotnet], [
if test "$ILASM" != ""; then
	mercury_cv_microsoft_dotnet="yes"
else
	mercury_cv_microsoft_dotnet="no"
fi
])
AC_MSG_RESULT([$mercury_cv_microsoft_dotnet])
ILASM=`basename "$ILASM"`
GACUTIL=`basename "$GACUTIL"`

# Check for an implementation of the Common Language Infrastructure.
AC_PATH_PROGS([CLI_INTERPRETER], [mono])

# Check for the C# (C sharp) compiler.
# csc is the Microsoft C# compiler.
# mcs is the Mono C# compiler targetting all runtime versions.
# (dmcs and gmcs are older aliases for the Mono C# compiler
# which we do not use.)

AC_CACHE_SAVE
case "$mercury_cv_with_csharp_compiler" in
    no)
        AC_MSG_ERROR(invalid option --without-csharp-compiler)
        exit 1
        ;;
    yes)
        AC_MSG_ERROR(missing argument to --with-csharp-compiler=... option)
        exit 1
        ;;
    "")
        CSC_COMPILERS="csc mcs"
        ;;
    *)
        CSC_COMPILERS="$mercury_cv_with_csharp_compiler"
        ;;
esac

AC_MSG_CHECKING([for a C sharp compiler])
AC_MSG_RESULT()
for CANDIDATE_CSC0 in $CSC_COMPILERS; do
    unset CANDIDATE_CSC
    unset ac_cv_path_CANDIDATE_CSC
    AC_CACHE_LOAD
    AC_PATH_PROG([CANDIDATE_CSC], [$CANDIDATE_CSC0])

    if test -z "$CANDIDATE_CSC"; then
        continue;
    fi
    CANDIDATE_CSC=`basename "$CANDIDATE_CSC"`

# Check that the compiler is suitable.
    case "$CANDIDATE_CSC" in
         csc*)
             # The Microsoft C# compiler and the Chicken Scheme compiler share
             # the same executable name, so if we find an executable named csc
             # above, check that it is actually the Microsoft C# compiler,
             # and if it is not, then try to use one of the other instead.
             $CANDIDATE_CSC 2>&1 | grep "^Microsoft" >/dev/null
             if test $? -ne 0
             then
                 AC_MSG_WARN([$CANDIDATE_CSC is not the Microsoft C sharp compiler])
                 continue;
             else
                 CSC="$CANDIDATE_CSC"
                 break;
             fi
        ;;

        *mcs)
            # We want to check that the 'mcs' compiler supports generics.
            # We test all the mono C sharp compilers in order to be more
            # defensive.
            AC_MSG_CHECKING([whether $CANDIDATE_CSC supports C sharp generics])

            cat > conftest.cs << EOF
    using System;
    using System.Collections.Generic;

    class Hello
    {
        private class ExampleClass { }

        static void Main()
        {
            Console.WriteLine("Hello world!");

            // Declare a list of type int.
            List<int> list1 = new List<int>();

            // Declare a list of type string.
            List<string> list2 = new List<string>();

            // Declare a list of type ExampleClass.
            List<ExampleClass> list3 = new List<ExampleClass>();
        }
    }
EOF
            echo $CANDIDATE_CSC conftest.cs >&AS_MESSAGE_LOG_FD 2>&1
            OUTPUT=$($CANDIDATE_CSC conftest.cs 2>&1)
            RESULT=$?
            rm -f conftest.cs conftest.exe
            echo $OUTPUT >&AS_MESSAGE_LOG_FD
            echo returned $RESULT >&AS_MESSAGE_LOG_FD
            if echo $OUTPUT | grep CS1644 > /dev/null; then
                # This compiler does not support generics.
                AC_MSG_RESULT(no)
                continue;
            elif test $RESULT -ne 0; then
                AC_MSG_RESULT(no)
                AC_MSG_WARN([$CANDIDATE_CSC returned exit code $RESULT])
                continue;
            else
                AC_MSG_RESULT(yes)
                CSC="$CANDIDATE_CSC"
                break;
            fi
        ;;

        *)
            CSC="$CANDIDATE_CSC"
            break;
        ;;
    esac
done

# If the user specified one or more compilers and we couldn't find any,
# then abort configuration.
if test "$mercury_cv_with_csharp_compiler" != "" -a "$CSC" = ""; then
    AC_MSG_ERROR([No suitable C sharp compiler could be found.])
    exit 1
fi

case "$CSC" in
    csc*)
        CSHARP_COMPILER_TYPE=microsoft
    ;;

    mcs*)
        CSHARP_COMPILER_TYPE=mono
    ;;

    *)
        CSHARP_COMPILER_TYPE=unknown
    ;;
esac

AC_SUBST([ILASM])
AC_SUBST([GACUTIL])
AC_SUBST([CSC])
AC_SUBST([CSHARP_COMPILER_TYPE])
AC_SUBST([CLI_INTERPRETER])
])

#-----------------------------------------------------------------------------#
#
# Java configuration.
#
AC_DEFUN([MERCURY_CHECK_JAVA],
[
# On Windows, the Java SDK has a high chance of being installed in a path
# containing spaces. The simplest solution is to keep only the basename.
# Everything will still work so long as the executables can be found on the
# PATH later.
AC_PATH_PROGS(JAVAC, javac)
case "$JAVAC" in
    *" "*)
        JAVAC=`basename "$JAVAC"`
        ;;
esac
AC_PATH_PROG(JAVA_INTERPRETER, java)
case "$JAVA_INTERPRETER" in
    *" "*)
        JAVA_INTERPRETER=`basename "$JAVA_INTERPRETER"`
        ;;
esac
AC_PATH_PROG(JAR, jar)
case "$JAR" in
    *" "*)
        JAR=`basename "$JAR"`
        ;;
esac

AC_CACHE_VAL(mercury_cv_java, [
if test "$JAVAC" != "" -a "$JAVA_INTERPRETER" != "" -a "$JAR" != ""; then
	AC_MSG_CHECKING(if the above Java SDK works and is sufficiently recent)
	cat > conftest.java << EOF
		// This program simply retrieves the constant
		// specifying the version number of the Java SDK and
		// checks it is at least 1.8, printing "Hello, world"
		// if successful.
		public class conftest {
		    public static void main (String[[]] args) {
			float	version;
			String	strVer = System.getProperty(
					"java.specification.version");

			try {
				version = Float.valueOf(strVer).floatValue();
			}
			catch (NumberFormatException e) {
				System.out.println("ERROR: \"java." +
						"specification.version\" " +
						"constant has incorrect " +
						"format.\nGot \"" + strVer +
						"\", expected a number.");
				version = 0f;
			}

			if (version >= 1.8f) {
				System.out.println("Hello, world\n");
			} else {
				System.out.println("Nope, sorry.\n");
			}
		    }
		}
EOF
	if
		echo "$JAVAC" conftest.java >&AS_MESSAGE_LOG_FD 2>&1 &&
		"$JAVAC" conftest.java >&AS_MESSAGE_LOG_FD 2>&1 &&
		echo "$JAVA_INTERPRETER" conftest > conftest.out 2>&AS_MESSAGE_LOG_FD &&
		CLASSPATH=. "$JAVA_INTERPRETER" conftest > conftest.out 2>&AS_MESSAGE_LOG_FD &&
		test "`tr -d '\015' < conftest.out`" = "Hello, world"
	then
		mercury_cv_java="yes"
	else
		mercury_cv_java="no"
	fi
	AC_MSG_RESULT($mercury_cv_java)
else
	if test "$JAVAC" = ""; then
		JAVAC="javac"
	fi
	if test "$JAVA_INTERPRETER" = ""; then
		JAVA_INTERPRETER="java"
	fi
	if test "$JAR" = ""; then
		JAR="jar"
	fi
	mercury_cv_java="no"
fi
])

AC_SUBST(JAVAC)
AC_SUBST(JAVA_INTERPRETER)
AC_SUBST(JAR)
])

AC_DEFUN([MERCURY_CHECK_JAVAC_HEAP_SIZE],
[
# The default maximum heap size is too small to build the standard library
# and other programs, so we need to increase it. The option to do that
# is non-standard, so we have to check that it is accepted.
AC_CACHE_VAL([mercury_cv_javac_flags_for_heap_size], [
if test "$mercury_cv_java" = "yes"
then
	AC_MSG_CHECKING([if the Java compiler accepts the max heap size option])
	# There are three versions of this. The '_test' version is the one
	# we test here; the '_mmake' version is the one that gets included
	# in the Mmake.vars file and the '_config' verison is the version
	# that gets included in the Mercury.config file.
	# The difference between the three versions is how much we need to
	# escape the double quotes; in particular we need to ensure that they
	# are *not* escaped in Mmake.vars and *are* escaped in Mercury.config.
	# The latter is necessary in order for --restricted-command-line to work.
	mercury_cv_javac_flags_for_heap_size_test="-J-Xmx256m"
	mercury_cv_javac_flags_for_heap_size_mmake="-J\"-Xmx1024m\""
	mercury_cv_javac_flags_for_heap_size_config="-J\\\"-Xmx1024m\\\""
	if "$JAVAC" "$mercury_cv_javac_flags_for_heap_size_test" -version \
                > /dev/null 2>&1
        then
		AC_MSG_RESULT(yes)
	else
		AC_MSG_RESULT(no)
		mercury_cv_javac_flags_for_heap_size=
	fi
else
	mercury_cv_javac_flags_for_heap_size=
fi
])
])

#-----------------------------------------------------------------------------#

# NOTE: updates to this macro may need to be reflected in compiler/globals.m.
# Generating an executable and running it does not work when cross-compiling.

AC_DEFUN([MERCURY_GCC_VERSION], [
AC_REQUIRE([AC_PROG_CC])

AC_MSG_CHECKING([what the gcc version is])

# -dumpfullversion has only been supported since GCC 7; if it is available we
# prefer its output since it is guaranteed to be in the form major.minor.patch.
# (It turns out that mingw32 gcc -dumpfullversion on Debian 12 reports
# "12-win32" anyway.)
raw_gcc_version=$($CC -dumpfullversion 2>/dev/null || $CC -dumpversion)

# The major version number should always be present.
# The minor version number and patchlevel are not always present.
# MinGW-w64 may add a suffix "-win32" or "-posix" that should be ignored.
#
# We use backticks for this command substitution because old versions of
# bash fail to parse this command with $() syntax.
# In particular, MacOS 14 and earlier use bash 3.2 as /bin/sh.
mercury_cv_gcc_version=`echo "${raw_gcc_version%-*}" | tr . ' ' | {
    read -r major minor patchlevel ignore;
    case $major in
	[[0-9]]*) ;;
	*) major=u ;;
    esac;
    case $minor in
	[[0-9]]*) ;;
	*) minor=u ;;
    esac;
    case $patchlevel in
	[[0-9]]*) ;;
	*) patchlevel=u ;;
    esac;
    echo "${major}_${minor}_${patchlevel}";
    }`

AC_MSG_RESULT([$mercury_cv_gcc_version])
])

#-----------------------------------------------------------------------------#

# Work out the C compiler type using a stronger test than AC_PROG_CC to
# distinguish between clang and gcc.

AC_DEFUN([MERCURY_CC_TYPE], [
AC_REQUIRE([AC_PROG_CC])
AC_MSG_CHECKING([what the C compiler type really is])

# MSVC uses different command line options to most other C compilers.
# Try to determine whether CC is MSVC based on the usage message.
#
$CC 2>&1 | grep "^Microsoft" >/dev/null
if test $? -eq 0
then
    cc_out_opt="-Fe"
    nologo_opt="-nologo"
else
    cc_out_opt="-o "
    nologo_opt=
fi

cat > conftest.c << EOF

#include <stdio.h>

int main(int argc, char **argv)
{
    #if defined(__clang__)
       printf("clang");
    #elif defined(__GNUC__)
       printf("gcc");
    #elif defined(_MSC_VER)
       #if defined(_M_IX86)
          printf("msvc_x86");
       #elif defined(_M_X64)
          printf("msvc_x64");
       #else
          printf("unknown");
       #endif
    #else
       printf("unknown");
    #endif

    return 0;
}
EOF

echo "$CC $nologo_opt ${cc_out_opt}conftest conftest.c" >&AS_MESSAGE_LOG_FD 2>&1
if
    # We direct the output to /dev/null because it appears to be
    # the only way to shut some C compilers up.
    $CC $nologo_opt ${cc_out_opt}conftest conftest.c 2>&1 > /dev/null
then
    AC_CACHE_VAL(mercury_cv_cc_type,
    mercury_cv_cc_type=`./conftest`)
else
    # This shouldn't happen as we have already checked for this.
    AC_MSG_ERROR([unexpected: $CC cannot create executable])
fi

AC_MSG_RESULT([$mercury_cv_cc_type])
])

#-----------------------------------------------------------------------------#

# NOTE: updates to this macro may need to be reflected in compiler/globals.m.
# Generating an executable and running it does not work when cross-compiling.
# Also, the Apple vendored version of clang "lies" about its version if you
# check using --version or by looking at __clang_major__ etc -- both of those
# return the XCode version instead of the underlying clang version.

AC_DEFUN([MERCURY_CLANG_VERSION], [
AC_REQUIRE([AC_PROG_CC])

AC_MSG_CHECKING([what the clang version is])

raw_clang_version=$($CC -dumpversion)

# The major version number should always be present.
# For GCC we allow for a suffix after the second or third number that should
# be ignored; it seems prudent to do the same for clang here as well.
mercury_cv_clang_version=`echo "${raw_clang_version%-*}" | tr . ' ' | {
    read -r major minor patchlevel ignore;
    case $major in
	[[0-9]]*) ;;
	*) major=u ;;
    esac;
    case $minor in
	[[0-9]]*) ;;
	*) minor=u ;;
    esac;
    case $patchlevel in
	[[0-9]]*) ;;
	*) patchlevel=u ;;
    esac;
    echo "${major}_${minor}_${patchlevel}";
    }`

AC_MSG_RESULT([$mercury_cv_clang_version])
])

#-----------------------------------------------------------------------------#

AC_DEFUN([MERCURY_MSVC_VERSION], [
AC_REQUIRE([AC_PROG_CC])

cat > conftest.c << EOF

#include <stdio.h>

int main(int argc, char **argv)
{

    printf("%d", _MSC_VER);
    return 0;
}
EOF

echo "$CC conftest.c -Fecontest" >&AS_MESSAGE_LOG_FD 2>&1
if
    $CC conftest.c -Feconftest > /dev/null
then
    mercury_cv_msvc_version=`./conftest`
else
    # This shouldn't happen as we have already checked for this.
    AC_MSG_ERROR([unexpected: $CC cannot create executable])
fi
])

#-----------------------------------------------------------------------------#
#
# Check if the C compiler targets x86.
# Note that checking $host is insufficient as we may be cross-compiling.
#

AC_DEFUN([MERCURY_CC_TARGETS_X86], [
AC_MSG_CHECKING(whether C compiler targets x86)

AC_CACHE_VAL(mercury_cv_cc_targets_x86,
    AC_COMPILE_IFELSE([AC_LANG_PROGRAM([[]], [[
            #ifdef __i386__
            #else
            #error "target is not x86"
            #endif
        ]])],
        [mercury_cv_cc_targets_x86=yes],
        [mercury_cv_cc_targets_x86=no])
)

AC_MSG_RESULT($mercury_cv_cc_targets_x86)
])

#-----------------------------------------------------------------------------#
#
# Check if the C compiler is configured to generate position-independent code.
# This may be the case even when not creating shared libraries.
#

AC_DEFUN([MERCURY_CC_GENERATES_PIC], [
AC_MSG_CHECKING(whether C compiler generates PIC)

AC_CACHE_VAL(mercury_cv_cc_generates_pic,
    AC_COMPILE_IFELSE([AC_LANG_PROGRAM([[]], [[
            #if defined(__PIC__) || defined(__pic__)
            #else
            #error "not PIC"
            #endif
        ]])],
        [mercury_cv_cc_generates_pic=yes],
        [mercury_cv_cc_generates_pic=no])
)

AC_MSG_RESULT($mercury_cv_cc_generates_pic)
])

#-----------------------------------------------------------------------------#
#
# Check if the POSIX threads library is pthreads-win32.
#

AC_DEFUN([MERCURY_HAVE_PTHREADS_WIN32], [
AC_MSG_CHECKING([if we are using pthreads-win32])

AC_COMPILE_IFELSE([AC_LANG_PROGRAM([[#include <pthread.h>]], [[
        #ifndef PTW32_VERSION
            #error I suppose not
        #endif
        int self_id(void)
        {
            return (int) pthread_self().p;
        }
    ]])],[mercury_cv_have_pthreads_win32=yes],[mercury_cv_have_pthreads_win32=no])

AC_MSG_RESULT($mercury_cv_have_pthreads_win32)
])

#-----------------------------------------------------------------------------#
